/* $Id: srcarb_intr.c,v 1.8 1998/07/20 22:40:42 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Mathiesen */

/* This file contains code showing how to set up SICL and an E1432 for
   interrupt-driven data collection. */

#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include "e1432.h"

#define	BLOCKSIZE	1024
#define	CHAN_MAX	32
#define	SRC_MAX		5

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

/* Make these global so the interrupt handler can get to them easily */
static E1432ID hw;
static SHORTSIZ16 chan_list[CHAN_MAX];
static SHORTSIZ16 group;
static int chan_count;

static SHORTSIZ16 src_list[SRC_MAX];
static SHORTSIZ16 src_group;
static int src_count;

static SHORTSIZ16 global_list[CHAN_MAX + SRC_MAX];
static SHORTSIZ16 global_group;
static int global_count;

static INST irq_id;
static int irq_laddr;
static int irq_flag;

/******************* arb source routines **************************/

static LONGSIZ32 src_buff[E1432_SRC_DATA_NUMWORDS_MAX];

#define SIG_LENGTH_MAX 100000

static LONGSIZ32 src_sigbuff[SIG_LENGTH_MAX];
static LONGSIZ32 *src_sigbuffptr;
static LONGSIZ32 *src_siglast;
static LONGSIZ32 arb_sigperiod;
static SHORTSIZ16 sigtype;
static LONGSIZ32 arb_siglength;

static int rd4data_set_count, rd4data_clr_count;
static SHORTSIZ16 arb_xfrmode, arb_xfrmodewait;

static SHORTSIZ16 source_mode, arb_mode;
static LONGSIZ32 arb_srcbufsize;
static LONGSIZ32 arb_xfrsize;

void
src_sig(LONGSIZ32 period, SHORTSIZ16 type)
{
    LONGSIZ32 i, src_arb_inc, src_arb_val;

#ifdef DPRINT
    (void) printf("src_sig: period = %d, type = %d\n",
	   period, type);
#endif

    if (arb_siglength > SIG_LENGTH_MAX)
	arb_siglength = SIG_LENGTH_MAX;
    switch (type)
    {
    case 1:
    default:
	/* sawtooth */
	src_arb_inc = -2 * ((LONGSIZ32) 0x80000000 / period);
	src_arb_val = 0;
	src_sigbuffptr = src_sigbuff;
	for (i = 0; i < arb_siglength; i++)
	{
	    *src_sigbuffptr = src_arb_val;
	    src_arb_val += src_arb_inc;
	    src_sigbuffptr = src_sigbuffptr + 1;
	}
    }
    src_sigbuffptr = src_sigbuff;
    src_siglast = &src_sigbuffptr[arb_siglength - 1];
}

void
xfr_src_sig(E1432ID lhw, SHORTSIZ16 chanID, LONGSIZ32 larb_xfrsize,
	    SHORTSIZ16 mode, LONGSIZ32 numwords)
{
    LONGSIZ32 i, words_left, xfrsize, maxtries;
    LONGSIZ32 *src_buffptr;

    /* copy larb_xfrsize words from src_sigbuff to src_buff, wrapping
       when needed */
#ifdef DPRINT
    (void) printf("xfr_src_sig: larb_xfrsize = %d, mode = %d, numwords = %d\n",
		  larb_xfrsize, mode, numwords);
#endif

    words_left = numwords;
    if (larb_xfrsize > E1432_SRC_DATA_NUMWORDS_MAX)
	larb_xfrsize = E1432_SRC_DATA_NUMWORDS_MAX;

    while (words_left > 0)
    {
	if (words_left < larb_xfrsize)
	    xfrsize = words_left;
	else
	    xfrsize = larb_xfrsize;
	src_buffptr = src_buff;
#ifdef DPRINT
	(void) printf("xfr_src_sig: src_sigbuffptr = 0x%x, src_siglast = "
		      "0x%x, words_left = %d, xfrsize = %d\n",
		      src_sigbuffptr, src_siglast, words_left,
		      xfrsize);
#endif
	for (i = 0; i < xfrsize; i++)
	{
	    *src_buffptr = *src_sigbuffptr;
#ifdef DPRINT1
	    (void) printf("xfr_src_sig: words_left = %d, "
			  "xfrsize = %d, i = %d\n",
			  words_left, xfrsize, i);
	    (void) printf("xfr_src_sig: * src_sigbuffptr = 0x%x, "
			  "src_sigbuffptr = 0x%x, src_siglast = 0x%x\n",
			  *src_sigbuffptr, src_sigbuffptr, src_siglast);
	    (void) printf("xfr_src_sig: * src_buffptr = 0x%x, "
			  "src_buffptr = 0x%x\n",
			  *src_buffptr, src_buffptr);
#endif
	    if (src_sigbuffptr >= src_siglast)
		src_sigbuffptr = src_sigbuff;
	    else
		src_sigbuffptr = src_sigbuffptr + 1;
	    src_buffptr = src_buffptr + 1;
	}

	/* write  xfrsize words to the substrate */

	i = 0;
	maxtries = 100000;

	for (i = 0; i < maxtries; i++)
	{
	    if (e1432_check_src_arbrdy(lhw, chanID, mode) == 1)
	    {
		rd4data_set_count = rd4data_set_count + 1;
		(void) e1432_write_srcbuffer_data(lhw, chanID,
						  src_buff,
						  (SHORTSIZ16) xfrsize,
						  mode);
		words_left = words_left - xfrsize;
		break;
	    }
	    else
		rd4data_clr_count = rd4data_clr_count + 1;
	}
	if (i == maxtries)
	{
	    (void) printf("xfr_src_sig:    write with wait, timed out \n");
	    return;
	}
    }
    words_left = numwords - words_left;
#ifdef DPRINT
    (void) printf("xfr_src_sig:    wrote %d to arb xfrbuffer\n",
		  words_left);
#endif
}


/******************* interrrupt routines **************************/

static int
read_data(void)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    int     i;

    /* Read some data */

    for (i = 0; i < chan_count; i++)
    {
/*
   (void) printf("irq handler input channel %d, chan_list[i](%x)\n", i,chan_list[i]);
 */
	CHECK(e1432_read_float64_data(hw, chan_list[i],
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, NULL, &count));
/*
   (void) printf("channel %dd    data points read =  %d\n", i , count);
 */
	if (count != BLOCKSIZE)
	{
	    DEBUG((void) printf("Actual count was %d\n", count));
	    return -1;
	}
    }

    return 0;
}

static int
srcstatus(void)
{
    int     i;

    /* 
       Only one interrupt per measurement for src_shutdown,
       src_overload and src_overread.  These src status conditions
       are latched until the source is turned off, then cleared. The
       source status irq is reenabled when the source is turned on.  */


    /* check source status */
    for (i = 0; i < src_count; i++)
    {
	if (e1432_check_src_shutdown(hw, src_list[i]) == 1)
	    (void) printf("    src_shutdown from  %d\n", src_list[i]);
	if (e1432_check_src_overload(hw, src_list[i]) == 1)
	    (void) printf("    src_overload from  %d\n", src_list[i]);
	if (e1432_check_src_overread(hw, src_list[i]) == 1)
	    (void) printf("    src_overread from %d\n", src_list[i]);
	if (e1432_check_src_overread(hw, src_list[i]) == 1)
	    (void) printf("    src_overread from %d\n", src_list[i]);
	if (e1432_check_src_arbrdy(hw, src_list[i], arb_xfrmode) == 1)
	{
#ifdef DPRINT
	    (void) printf("    src_arbrdy from %d\n", src_list[i]);
#endif
	    /* down load signal */
	    xfr_src_sig(hw, src_list[i], arb_xfrsize, arb_xfrmode,
			arb_xfrsize);
	}
    }

    return 0;
}

#ifdef DPRINT2
static void
irqstatus(void)
{
    LONGSIZ32 status;
    LONGSIZ32 xfrbuf_wds, srcbuf_state_AB, srcbuf_state_A, srcbuf_state_B;

    /* check irq status */

    if (src_count > 0)
    {
	CHECK(e1432_read32_register(hw, src_list[0],
				    E1432_IRQ_STATUS2_REG, &status));
	(void) printf("E1432_IRQ_STATUS2_REG = 0x%x\n", status);
	CHECK(e1432_read32_register(hw, src_list[0],
				    E1432_SRC_STATUS_REG, &status));
	(void) printf("E1432_SRC_STATUS_REG = 0x%x\n", status);
    }
    else
    {
	if (chan_count > 0)
	{
	    CHECK(e1432_read32_register(hw, chan_list[0],
					E1432_IRQ_STATUS2_REG,
					&status));
	    (void) printf("    E1432_IRQ_STATUS2_REG = 0x%x\n\n",
			  status);
	}
    }

    (void) e1432_get_src_arbstates(hw, src_list[0], &xfrbuf_wds,
				   &srcbuf_state_AB, &srcbuf_state_A,
				   &srcbuf_state_B);
    (void) printf("xfrbuf_wds = %d, srcbuf_state_AB = 0x%x, "
		  "srcbuf_state_A = 0x%x, srcbuf_state_B = 0x%x\n",
		  xfrbuf_wds, srcbuf_state_AB,
		  srcbuf_state_A, srcbuf_state_B);

    return 0;
}
#endif

static void
irq_handler(INST id, long reason, long sec)
{
#ifdef DPRINT2
    DEBUG((void) printf("\nirq_handler called: sec = 0x%x\n", sec));
#endif

    if (id != irq_id)
	(void) printf("Error: irq_handler got wrong id\n");
    if (reason != I_INTR_VXI_SIGNAL)
	(void) printf("Error: irq_handler got wrong reason\n");
    if ((sec & E1432_IRQ_STATUS_LADDR_MASK) != irq_laddr)
	(void) printf("Error: irq_handler got wrong laddr\n");

    irq_flag = 1;

    /* print out status */
#ifdef DPRINT2
    irqstatus();
#endif

    if ((sec & E1432_IRQ_SRC_STATUS) != 0)
	(void) srcstatus();
    if ((sec & E1432_IRQ_BLOCK_READY) != 0)
	(void) read_data();
    if ((sec & (E1432_IRQ_BLOCK_READY | E1432_IRQ_SRC_STATUS)) == 0)
	(void) printf("Error: irq_handler got wrong sec\n");

    /* reset interrupt */

    if (src_count > 0)
	(void) e1432_reenable_interrupt(hw, src_group);
    else if (chan_count > 0)
	(void) e1432_reenable_interrupt(hw, group);
}


static int
irq_setup(int laddr, int *line)
{
    struct vxiinfo info;
    unsigned long slot0_laddr;
    char    addr[16];
    INST    id;
    int     i;

    /* Get the interrupt line to use */
    id = iopen("vxi");
    if (id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", id));
	return -1;
    }
    CHECK(ivxibusstatus(id, I_VXI_BUS_LADDR, &slot0_laddr));
    CHECK(ivxirminfo(id, slot0_laddr, &info));
    for (i = 0; i < 8; i++)
	if (info.int_handler[i] != -1)
	{
	    *line = info.int_handler[i];
	    break;
	}
    if (i == 8)
    {
	DEBUG((void) printf("Error: no interrupt lines available\n"));
	return -1;
    }
    DEBUG((void) printf("Using VME interrupt line %d\n", *line));

    /* Set up the interrupt handler routine */
    irq_laddr = laddr;
    (void) sprintf(addr, "vxi,%d", irq_laddr);
    irq_id = iopen(addr);
    if (irq_id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", irq_id));
	return -1;
    }
    CHECK(ionintr(irq_id, irq_handler));
    CHECK(isetintr(irq_id, I_INTR_VXI_SIGNAL, 1));

    return 0;
}

/**********************************************************************/

int
main(void)
{
    int     i, line,error;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 laddr = 8;

    
    /* Initialize the library */
    CHECK(e1432_init_io_driver());

    /* Change this 0 to 1 to see call tracing */
    e1432_trace_level(0);

    /* Use e1432_get_hwconfig to see if the module already has
       firmware.  If this errors, assume we need to install firmware,
       so use e1432_install to do it. */
    error = e1432_get_hwconfig(1, &laddr, &hwconfig);
    if (error != 0)
    {
	(void) printf("downloading /opt/e1432/lib/sema.bin\n");
        error = e1432_install(1, &laddr, 0,
                              "/opt/e1432/lib/sema.bin");
        if (error)
	{
            (void) printf("e1432_install failed, error: %d\n", error);
	    return -1;
	}
	
	CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    }

    /* Turn on debugging prints, a good idea while developing.  But do
       it after the above e1432_get_hwconfig, because we know that
       might error if the firmware is not yet installed in the
       module. */
    (void) e1432_print_errors(1);

    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    /* Create channel group */
    if (hwconfig.input_chans > 0)
    {
	chan_count = hwconfig.input_chans;
	if (chan_count > CHAN_MAX)
	    chan_count = CHAN_MAX;
	for (i = 0; i < chan_count; i++)
	    chan_list[i] = E1432_INPUT_CHAN(i + 1);
	group = e1432_create_channel_group(hw, chan_count, chan_list);
	if (group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				group));
	    return -1;
	}
    }
    else
	chan_count = 0;

    /* Create source group */
    if (hwconfig.source_chans > 0)
    {
	src_count = hwconfig.source_chans;
	if (src_count > SRC_MAX)
	    src_count = SRC_MAX;
	for (i = 0; i < src_count; i++)
	    src_list[i] = E1432_SOURCE_CHAN(i + 1);
	src_group = e1432_create_channel_group(hw, src_count, src_list);
	if (src_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				src_group));
	    return -1;
	}
    }
    else
	src_count = 0;

    /* Create global group  of all input and source channels */

    global_count = 0;

    for (i = 0; i < chan_count; i++)
    {
	global_list[global_count] = chan_list[i];
	global_count = global_count + 1;
    }
    for (i = 0; i < src_count; i++)
    {
	global_list[global_count] = src_list[i];
	global_count = global_count + 1;
    }

    if (global_count > 0)
	global_group = e1432_create_channel_group(hw, global_count,
						  global_list);

    (void) printf("# input channels = %d, # source channels = %d\n",
		  chan_count, src_count);
    (void) printf("# total channels = %d\n", global_count);


/* initialize source */
    if (src_count > 0)
    {

	source_mode = E1432_SOURCE_MODE_ARB;
	arb_mode = E1432_SRCBUFFER_CONTINUOUS;
	arb_srcbufsize = 4096;

	arb_siglength = 16384;
	arb_sigperiod = 256;
	sigtype = 1;

	arb_xfrsize = arb_srcbufsize;
	arb_xfrmode = E1432_SRC_DATA_MODE_AB;
	arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITAB;

	(void) printf("arb_xfrsize %d, arb_srcbufsize %d, arb_siglength %d\n",
		      arb_xfrsize, arb_srcbufsize, arb_siglength);
	(void) printf("arb_xfrmode 0x%x, arb_xfrmodewait %d\n",
		      arb_xfrmode, arb_xfrmodewait);

	CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));
	CHECK(e1432_set_source_mode(hw, src_group, source_mode));
	CHECK(e1432_set_ramp_rate(hw, src_group, 0));
	CHECK(e1432_set_range(hw, src_group, 1.0));
	CHECK(e1432_set_amp_scale(hw, src_group, .9995));

	CHECK(e1432_set_srcbuffer_mode(hw, src_group, arb_mode));
	CHECK(e1432_set_srcbuffer_size(hw, src_group, arb_srcbufsize));
	CHECK(e1432_set_srcbuffer_init(hw, src_group,
				       E1432_SRCBUFFER_INIT_EMPTY));

/** create signal array ***/
	src_sig(arb_sigperiod, sigtype);

	/* down load signal */
	xfr_src_sig(hw, src_list[0], arb_xfrsize, arb_xfrmodewait,
		    arb_xfrsize);
	if (e1432_check_src_arbrdy(hw, src_list[0], arb_xfrmodewait))
	{
	    /* load B as well as A */
	    xfr_src_sig(hw, src_list[0], arb_xfrsize, arb_xfrmodewait,
			arb_xfrsize);
	}
	(void) printf("done pre-loading buffer\n");

	/* Set up interrupts */
	CHECK(irq_setup(laddr, &line));
	CHECK(e1432_set_interrupt_priority(hw, src_group, line));
	CHECK(e1432_set_interrupt_mask(hw, src_group,
				       E1432_IRQ_BLOCK_READY |
				       E1432_IRQ_SRC_STATUS));
	CHECK(e1432_set_blocksize(hw, src_group, BLOCKSIZE));
    }
    else
    {
	/* Set up interrupts */
	CHECK(irq_setup(laddr, &line));
	CHECK(e1432_set_interrupt_priority(hw, group, line));
	CHECK(e1432_set_interrupt_mask(hw, group,
				       E1432_IRQ_BLOCK_READY |
				       E1432_IRQ_SRC_STATUS));

	CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    }


    /* Start measurement */
    (void) printf("starting measurement\n");
    CHECK(e1432_init_measure(hw, global_group));

    /* Sleep for a long time.  Note that receiving an interrupt
       aborts the sleep, so we must keep calling it. */
    for (;;)
    {
	irq_flag = 0;
	(void) sleep(1);
/*      if (irq_flag == 0) irqstatus();
 */
    }


    /* NOTREACHED */
    return 0;
}
